<?php
/**
 * Moore's CustomZ Theme Auto-Updater
 *
 * Checks sentrasys.dev/api/sentra-repo for new theme versions and lets WordPress
 * update the theme automatically through the standard Themes update mechanism.
 *
 * Endpoint used:
 *   GET /api/sentra-repo/public/latest/{app_id}/{channel}
 *   GET /api/sentra-repo/download/{version_id}   (served by sentra-core)
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MooresCustomZ_Theme_Updater {

    /** App ID registered in the Sentra Repo for this theme */
    const APP_ID  = 'moorescustomz';

    /** Release channel to track */
    const CHANNEL = 'stable';

    /** Manifest endpoint base URL — no trailing slash */
    const REPO_BASE = 'https://sentrasys.dev/api/sentra-repo';

    /** WordPress theme slug (matches the theme folder name) */
    const THEME_SLUG = 'moorescustomz';

    /** How long (seconds) to cache the remote version check */
    const CACHE_TTL = 43200; // 12 hours

    /** Transient key for cached manifest data */
    const TRANSIENT_KEY = 'moorescustomz_update_manifest';

    public function __construct() {
        add_filter( 'pre_set_site_transient_update_themes', [ $this, 'inject_update' ] );
        add_filter( 'themes_api',                           [ $this, 'theme_info' ], 10, 3 );
        add_action( 'upgrader_process_complete',            [ $this, 'purge_cache' ],  10, 2 );
    }

    // -------------------------------------------------------------------------
    // Fetch & cache the remote manifest
    // -------------------------------------------------------------------------

    private function get_manifest(): ?array {
        $cached = get_site_transient( self::TRANSIENT_KEY );
        if ( is_array( $cached ) && ! empty( $cached['version'] ) ) {
            return $cached;
        }

        $url = self::REPO_BASE . '/public/latest/' . self::APP_ID . '/' . self::CHANNEL;

        $theme   = wp_get_theme( self::THEME_SLUG );
        $wp_ver  = get_bloginfo( 'version' );
        $thm_ver = $theme->get( 'Version' ) ?: '2.0.0';

        $response = wp_remote_get( $url, [
            'timeout'    => 10,
            'user-agent' => 'WordPress/' . $wp_ver . '; MooresCustomZ-Updater/' . $thm_ver,
            'headers'    => [
                'Accept' => 'application/json',
            ],
        ] );

        if ( is_wp_error( $response ) ) {
            return null;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code !== 200 ) {
            return null;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! is_array( $data ) || empty( $data['version'] ) ) {
            return null;
        }

        // Normalise download URL: if relative, make it absolute
        if ( ! empty( $data['download_url'] ) && strpos( $data['download_url'], 'http' ) !== 0 ) {
            $data['download_url'] = 'https://sentrasys.dev' . $data['download_url'];
        }

        set_site_transient( self::TRANSIENT_KEY, $data, self::CACHE_TTL );

        return $data;
    }

    // -------------------------------------------------------------------------
    // Hook: inject into WordPress theme update transient
    // -------------------------------------------------------------------------

    public function inject_update( $transient ) {
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        $manifest = $this->get_manifest();
        if ( ! $manifest ) {
            return $transient;
        }

        $remote_version  = $manifest['version'];
        $current_version = $transient->checked[ self::THEME_SLUG ] ?? '2.0.0';

        if ( version_compare( $remote_version, $current_version, '>' ) ) {
            $transient->response[ self::THEME_SLUG ] = [
                'theme'       => self::THEME_SLUG,
                'new_version' => $remote_version,
                'url'         => $manifest['homepage'] ?? 'https://sentrasys.dev',
                'package'     => $manifest['download_url'],
                'requires'    => $manifest['requires']     ?? '6.1',
                'requires_php'=> $manifest['requires_php'] ?? '7.4',
            ];
        }

        return $transient;
    }

    // -------------------------------------------------------------------------
    // Hook: supply theme info for the "View details" pop-up
    // -------------------------------------------------------------------------

    public function theme_info( $result, string $action, $args ) {
        if ( $action !== 'theme_information' ) {
            return $result;
        }

        if ( ! isset( $args->slug ) || $args->slug !== self::THEME_SLUG ) {
            return $result;
        }

        $manifest = $this->get_manifest();
        if ( ! $manifest ) {
            return $result;
        }

        $changelog = $manifest['changelog'] ?? $manifest['notes'] ?? '';

        return (object) [
            'name'          => "Moore's CustomZ",
            'slug'          => self::THEME_SLUG,
            'version'       => $manifest['version'],
            'author'        => '<a href="https://sentrasys.dev">Sentra Systems</a>',
            'homepage'      => $manifest['homepage']    ?? 'https://sentrasys.dev',
            'requires'      => $manifest['requires']    ?? '6.1',
            'tested'        => $manifest['tested']      ?? '',
            'requires_php'  => $manifest['requires_php'] ?? '7.4',
            'download_link' => $manifest['download_url'],
            'last_updated'  => $manifest['last_updated'] ?? '',
            'sections'      => [
                'description' => "Moore's CustomZ — custom WordPress theme built on the Sentra platform.",
                'changelog'   => $changelog ?: 'See <a href="https://sentrasys.dev">sentrasys.dev</a> for release notes.',
            ],
        ];
    }

    // -------------------------------------------------------------------------
    // Hook: purge cache after an upgrade so next check is fresh
    // -------------------------------------------------------------------------

    public function purge_cache( $upgrader, array $hook_extra ): void {
        if ( isset( $hook_extra['theme'] ) && $hook_extra['theme'] === self::THEME_SLUG ) {
            delete_site_transient( self::TRANSIENT_KEY );
        }
    }
}
